//
//  ViewModifiers.swift
//  Do It
//
//  Created by Jim Dovey on 2/4/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI
import Combine

struct DoubleShadow: ViewModifier {
    private var radius: CGFloat
    
    init(_ radius: CGFloat = 10.0) {
        self.radius = radius
    }
    
    func body(content: Content) -> some View {
        content
            .shadow(color: Color.black.opacity(0.1),
                    radius: radius, x: 0, y: radius * 1.2)
            .shadow(color: Color.black.opacity(0.2),
                    radius: max(radius/10, 1), x: 0, y: 1)
    }
}

struct BorderedTextField: ViewModifier {
    func body(content: Content) -> some View {
        content
            .multilineTextAlignment(.center)
            .padding(.vertical, 12)
            .background(
                RoundedRectangle(cornerRadius: 10, style: .continuous)
                    .foregroundColor(Color(UIColor.tertiarySystemFill))
            )
    }
}

// START:HoverableModifier
struct NicelyHoverable: ViewModifier {
    private let padding: EdgeInsets
    private let effect: HoverEffect
    // END:HoverableModifier
    
    // START:PaddingInit
    init(_ padding: CGFloat = 8, _ edges: Edge.Set = .all,
         _ effect: HoverEffect = .automatic) {
        self.padding = EdgeInsets(
            top: edges.contains(.top) ? padding : 0,
            leading: edges.contains(.leading) ? padding : 0,
            bottom: edges.contains(.bottom) ? padding : 0,
            trailing: edges.contains(.trailing) ? padding : 0
        )
        self.effect = effect
    }
    // END:PaddingInit
    // START:HoverableModifier
    
    init(_ insets: EdgeInsets, _ effect: HoverEffect = .automatic) {
        self.padding = insets
        self.effect = effect
    }
    
    func body(content: Content) -> some View {
        content
            .padding(padding)
            .hoverEffect(effect)
            .padding(-padding)
    }
}
// END:HoverableModifier

// START:SceneDismissalModifier
struct SceneSessionDismissal: ViewModifier {
    private let session: UISceneSession
    private let errorHandler: ((Error) -> Void)?
    
    init(_ session: UISceneSession, errorHandler: ((Error) -> Void)? = nil) {
        self.session = session
        self.errorHandler = errorHandler
    }
    
    // START:SceneDismissalBody
    func body(content: Content) -> some View {
        // END:SceneDismissalBody
        // <literal:elide> ... </literal:elide>
        // END:SceneDismissalModifier
        // START:SceneDismissalBody
        let button = Button(action: {
            UIApplication.shared
                .requestSceneSessionDestruction(self.session, options: nil,
                                                errorHandler: self.errorHandler)
        }) {
            Text("Done")
                .bold()
        }
        // END:SceneDismissalBody
        .niceHoverEffect()
        // START:SceneDismissalBody
        
        return NavigationView {
            content.navigationBarItems(leading: button)
        }
        .navigationViewStyle(StackNavigationViewStyle())
        // START:SceneDismissalModifier
    }
    // END:SceneDismissalBody
}
// END:SceneDismissalModifier

// START:SceneDismissalHelper
extension View {
    // END:SceneDismissalHelper
    func doubleShadow(radius: CGFloat = 10) -> some View {
        modifier(DoubleShadow(radius))
    }
    
    // START:HoverFn
    func niceHoverEffect(_ padding: CGFloat = 8, _ edges: Edge.Set = .all,
                         _ effect: HoverEffect = .automatic) -> some View {
        modifier(NicelyHoverable(padding, edges, effect))
    }
    
    func niceHoverEffect(_ insets: EdgeInsets, _ effect: HoverEffect = .automatic) -> some View {
        modifier(NicelyHoverable(insets, effect))
    }
    // END:HoverFn
    
    // START:HoverIconFn
    func iconHoverEffect(_ color: Color = Color(.systemGroupedBackground),
                         _ padding: CGFloat = 8,
                         _ effect: HoverEffect = .automatic) -> some View {
        let insets = EdgeInsets(top: padding+6, leading: padding,
                                bottom: padding+6, trailing: padding)
        return modifier(NicelyHoverable(insets, effect))
    }
    // END:HoverIconFn
    
    // START:SceneDismissalHelper
    func dismissingSceneSession(
        _ session: UISceneSession,
        errorHandler: ((Error) -> Void)? = nil
    ) -> some View {
        modifier(SceneSessionDismissal(session, errorHandler: errorHandler))
    }
}
// END:SceneDismissalHelper

extension Text {
    func filledPlatter() -> some View {
        modifier(BorderedTextField())
    }
}

extension TextField {
    func filledPlatter() -> some View {
        modifier(BorderedTextField())
    }
}

struct ViewModifiers_Previews: PreviewProvider {
    static var _sampleText: String = "Sample Text"
    static var sampleText: Binding<String> = Binding(
        get: { _sampleText }, set: { _sampleText = $0 })

    static var previews: some View {
        Group {
            Circle()
                .frame(width: 300, height: 300)
                .foregroundColor(.white)
                .modifier(DoubleShadow())

            Circle()
                .frame(width: 300, height: 300)
                .foregroundColor(.white)
                .modifier(DoubleShadow(20))

            Circle()
                .frame(width: 300, height: 300)
                .foregroundColor(.white)
                .modifier(DoubleShadow(6))

            Text("Start Making Things")
                .padding(.horizontal)
                .modifier(BorderedTextField())

            TextField("Title", text: sampleText)
                .modifier(BorderedTextField())
            
            Button(action: {}) {
                Text("Button").bold()
            }
            .niceHoverEffect()
            
            Button(action: {}) {
                Image(systemName: "info.circle.fill")
                    .imageScale(.large)
            }
            .iconHoverEffect()
            
            Text("Highlight Effect")
                .padding()
                .hoverEffect(.highlight)
            
            Text("Lift Effect")
                .padding()
                .hoverEffect(.lift)
        }
        .previewLayout(.fixed(width: 350, height: 350))
    }
}
